#! /bin/nawk -f
# idlekiller
# P. Farrell, 8 Mar 1993
#  Modified  25 Mar 1993 - Add program exemption list
# Uses "new" (1985) awk interpreter, called /bin/nawk on Ultrix but
# possibly just "awk" on other systems.
#
# Script to find and kill login processes that have been idle more than
# number of hours specified as argument (default = 4 if no argument).
# Nevers kills processes belonging to list of privileged users (below).
# Also, even if user/tty shows idle, does not kill process on that tty if
# any process running on that tty is in a list of exempt programs (below).
# Exit code is number of idle users/ttys found.
# 
# Must be run from the "root" account.  NOT a daemon - run once each time
# you want to kill idle jobs (e.g., run every hour from crontab).
# Uses "ps" and "w" commands and is highly sensitive to exact format
# of their output. If your commands do not match output formats of
# samples shown below, you will have to adjust various indexing functions
# in this script.
#
# Sample output from "ps augx" command:
#         1         2         3         4         5         6
#123456789012345678901234567890123456789012345678901234567890
#USER       PID %CPU %MEM   SZ  RSS TT STAT   TIME COMMAND
#farrell  18525 29.1  0.8  992  848 se R      0:01 ps augx
#mehedi   16716 20.7 14.57292416176 sb R      5:13 a.out
#davidc   17310 18.0  0.2  244  164 q7 R      1:02 gepinv
#alexa    17908  1.2  0.3  628  340 ?  S      0:10 /local/cap/bin/aufs -n pangea
#
# Sample output from "w" command:
#         1         2         3         4         5         6
#123456789012345678901234567890123456789012345678901234567890
# 11:25am  up 35 days,  3:54,  106 users,  load average: 11.48, 11.35, 10.08
#User     tty from           login@  idle   JCPU   PCPU  what
#packwood t7 Tip-SRB.Stanfo  9:27am                      vi lh.tex
#towle    t9 Mitchell-MPG1p  9:29am            2      2  rn
#jouaux   ta artemis:0.0    24Feb9310days                -
#aziz     tb peteng-pc15.St Fri 9am  1:29     13         -csh
#deniz    td Tip-PetEngA.St  9:31am     7      4         -csh
#
# All the work is done in the BEGIN and END sections, because this
# script has no input file!

BEGIN {
# Set timeout:  number of hours user can stay idle before he is logged out.
# If there is an argument to this script, use it as the timeout value.
# Otherwise, use default value 4 hours.
    timeout=4  # default 4 hours
    if (ARGC > 1) {	# always 1 arg - name of program is ARGV[0]
        timeout=ARGV[1]+0	# adding zero forces numeric conversion
        ARGV[1]=""		# set null so not intepreted as filename
	if (timeout <= 0 ) timeout=4   # invalid value, force default
        }  # end of test of arguments.

# The list of users who are never logged out.  Every element in the list
# MUST be surrounded by blanks!  (To properly test whole usernames against
# the list elements).
    userskiplist = " root farrell kai norm "
# The list of programs that are never logged out.  Every element in the list
# MUST be surrounded by blanks!  (To properly test whole usernames against
# the list elements).  These are programs that must run connected to a
# terminal for output, but can run for many hours with no other activity
# on that terminal.  They would appear in the idle list even though the
# terminal is active.  Any program to be exempted in this way from the
# idle killing action must be run so that the program is called by this
# exact name.  Only the first word of the command line string is checked
# to see if it matches one of these exempt programs.
    progskiplist = " gepinv "

# Initialize list of user names that were idle
    idlelist = " "

# Get the output from the "w" command.  Discard first two header lines.
# Discard all logins that fail the idle test.  Skip privileged users.
    nidle=0
    nread=0
    while ("/usr/ucb/w" | getline test) {
        nread++
        if (nread < 3) continue     # skip header lines
# Test is whether idle longer > 4 hours.  Parse out the idle time.
# If it contains the string "days" it is > 24 hours.  If it has
# a colon, > 1 hour.  The idle time field can be blank,
# or run together with previous one in extreme cases, so normal splitting
# by blanks will not get it correct in all cases.  Instead, use actual
# column positions to put the portion that is normally the "hours" 
# (before the colon) into a variable.  If idle > 24 hours, this will
# contain the "d" from "days"; convert to hours.
        idletime=substr(test,35,3)
        if (substr(idletime,3) == "d") idletime=24*substr(idletime,1,2)
        idletime+=0    # to force conversion of leading blank to numeric 0.
        idlemin=substr(test,39,2)+0   # adding zero forces conversion to numeric
        idletime+=idlemin/60
#        print "user: " substr(test,1,50) "  idletime: " idletime  #debug
        if (idletime < timeout ) continue    # not idle long enough
#        print "idle user: " test  #debug
#        print "idle time: " idletime #debug
# skip the "privileged" users who never get logged out.
        split (test,testarr)        # split line from "w" on blanks
        if (match(userskiplist," " testarr[1] " ") > 0) continue
# Now have an idle user.  Save his username and tty line in an array.
# Also save simple list of all idle users in one variable
        idlelist=idlelist testarr[1] " "
        nidle++
        idlename[nidle]=testarr[1]
        idletty[nidle]=testarr[2]
        }   # end of while loop reading output from "w"

# Have read process list and idle user list into arrays.  Quit to END
# section.
    exit
} # end of BEGIN section

# Find the processes associated with each idle tty line and kill them.
# Note that can have multiple entries for same person, each on a 
# different tty line, so need two stage matching process.
END {
# If no idle processes, just quit
    if (nidle == 0) exit 0

# Get a listing of all running processes.  Do it just once.
# Have the list sorted by username, and then in reverse process id number
# per username.  This processing will make matching easier later.
# Also, generally want to kill child processes before parent.
# Only save lines for people in the idle list; may or may not be for
# the idle tty line of that person (to be checked below).
    nprocs=0
    while ("/bin/ps auxg | sort +0 -1 +1nr -2" | getline test) {
        split (test,testarr," ")        # split line from "ps" on blanks
        if (match(idlelist," " testarr[1] " ") == 0) continue  # not idle user
#        print "possible process:" test  #debug
        nprocs++
        process[nprocs]=test
        }  # end of loop to get process list

# Loop over the list of saved users/ttys that are idle.
# For each user/tty, find running processes that match.  Because fields
# in ps output can run into each other, cannot match tty based on
# blank-delimiters. Instead, have to match according to column positions.
    for (i=1; i<=nidle; i++) {
#        print "idle person:" idlename[i],idletty[i]  #debug
        pidlist = " "   # initialize matching pid list for this login
# Compare to list of saved processes that are possible match candidates.
# Check if possible match candidate is an "exempt" process; if so,
# set a flag so login on this tty will not be killed
        pef=0
        for (j=1; j<=nprocs && pef==0; j++) {
            if (match(process[j],"^" idlename[i] " ") == 0) continue # not user
            if (match(substr(process[j],36,2),idletty[i]) == 0) continue #nottty
# Found a process for this user/tty combination.
# Is this an "exempt" program?  If so, stop checking for this user/tty combo.
            split(substr(process[j],51),testarr," ")
            if (match(progskiplist," " testarr[1] " ") > 0) {
                pef=1
#                print "exempt program: " process[j]  #debug
                continue  # back to loop test for process list loop
            }  # this program exempted, skip this user/tty combo.
# Add process id to a list being created for this person.
#            print "match idle:" process[j]  #debug
            pidlist=pidlist substr(process[j],10,5) " "
            }  # end of loop over process list to check against current user

# If any matching pids were found for this user/tty, and program on that
# tty was not exempted, execute kill command
# for them.   They will be in reverse pid number order, which is best.
#        print "match pidlist:" pidlist  #debug
        if (pidlist != " " && pef == 0) system("kill -9" pidlist)
#        if (pidlist != " " && pef == 0) system("echo kill -9" pidlist)  #debug
        }  # end of loop over saved idle users/ttys.

    exit (nidle)    # exit code contains number of idle sessions killed
} # end of END section
