/*
 * graph - create XY data plots 
 *
 * Authors: Dave Hale, 11/2/82
 *          Glenn Kroeger 12/25/83
 *
 * Log scales added by Chuck Karish, 7/5/85, based on a preliminary
 *          implementation by an anonymous petroleum engineer.
 *
 * GKS converted by Glenn Kroeger  8/15/85
 *
 * 1/25/86: Added loop to ignore comment lines and parameter lines
 *          which begin with non-digits, to allow embedded parameters
 *          and comments at breaks between plots.  Also added
 *          "error=" parameter, and error catching for
 *          window->ur.[xy] == window->ll.[xy].               --crk
 *
 * 3/5/86:      Added test for d[xy]num <= 0 to avoid infinite loops. --crk
 *
 * 10/10/86:    Call GKS to open the terminal as a workstation, if
 *               output is not redirected.  Depends on /etc/ttywstypes
 *               or WSTYPE environment variable.               --crk
 */
char *documentation[] = {
" NAME",
"     graph - create XY data plots (outputs VPLOT)",
" SYNOPSIS",
"     graph [data reading options] [data display options] [axis options]",
" OPTIONS",
"     data reading     |   data display       |    axes ",
"    ------------------|----------------------|---------------------",
"     x=string         |    symbol=string     | xinch,yinch=float",
"     format=string    |    symbscale=float   | xmin,xmax,ymin,ymax=float",
"     in=filename      |    plotfat=int       | dxnum,dynum=float",
"     out=filename     |    plotcol=int       | x0num,y0num=float",
"     nplot=int        |    linetype=int      | nxtic,nytic=int",
"                      |    logx,logy=int     | box,grid=int",
"                      |    loglog=int        | axiscol,axisfat=int",
"                      |                      | gridcol,gridfat=int",
"                      |                      | title,xlabel,ylabel=string",
"                      |                      | titlsz,lablsz,numsz=float",
"                      |                      | titlcol,lablcol,numcol=int",
"                      |                      | titlfat,lablfat,numfat=int",
"From Ethernet terminals, 'setenv WSTYPE terminal type' (graphon, tek4107, etc.)",
" SEE ALSO",
"     graph(L), gks(L), vplot(L), /etc/ttywstypes"
};
int     doclength = { sizeof documentation/sizeof documentation[0] };

#include <stdio.h>
#include <math.h>
#include <sys/ioctl.h>
#include <sgtty.h>
#include "gks.h"
#include "device.h"

#define NPMAX 1000
#define WORKSTATION  1
#define USER  1
#define INCH  2

int xargc; 
char **xargv;
struct sgttyb ttystat;
char name[40];

main (argc,argv)
int argc; 
char **argv;
{
	register int i,j;
	int npts[NPMAX],nplot,nxtic,nytic,nchr,nychrmax;
	int symbol[NPMAX],plotcol[NPMAX],plotfat[NPMAX],linetype[NPMAX];
	int axiscol[2],axisfat[2];
	int titlcol,titlfat,lablcol,lablfat,numcol,numfat;
	int grid,box,ascii,increment,gridcol[2],gridfat[2];
	int piped_in, piped_out;
	int x_start,x_end,cycle_index,cycle_range,logx,x_tic,errflag,k;
	int y_start,y_end,logy,y_tic,ntic,loglog;

	float titlsz,lablsz,numsz,xscale,yscale;
	float symbsc[NPMAX];
	float xmin,xmax,ymin,ymax;
	float dx,x0,xinch,yinch,xneeded,yneeded;
	float dxtic,dytic,x0tic,y0tic,dummy;
	float xf,dxnum,dxbuf,x0num,dynum,dybuf,y0num,xnum,ynum,xtic,ytic;
	float xorig,yorig;
	float *farray,*fptr;
	float pagelength,ticlength;

	char chr,xlabel[80],ylabel[80],title[80];
	char *sptr,string[80],word[80];

	FILE *in_file,*out_file;

	register Gpoint *pptr;
	Gpoint *points[NPMAX],tick1[2],tick2[2];
	Gpoint vector;
	Gpoint *vec= &vector;
	Glimit wind,pg;
	Glimit *window = &wind;
	Glimit *page = &pg;
	Gtxalign algn;
	Gtxalign *align = &algn;
	Gtxfp fntpr;
	Gtxfp *fontprec = &fntpr;
	int wstype;
		char *typename, *ggetwstype();
	Gipoint rastersize;
	Gpoint dcsize;
		Gqloc dummy2;
	int dcunits;

	/*
	 * get parameters, open files, and set defaults
	 */
	strcpy(name,"graph");
	xargc = argc; 
	xargv=argv;

	/* 
	 * If no arguments, and not in a pipeline, self document
	 */
	piped_in = ioctl ((fileno (stdin)), TIOCGETP, &ttystat);
	piped_out = ioctl ((fileno (stdout)), TIOCGETP, &ttystat);
	if (argc == 1 && !piped_in)
	{
		for( i=0; i<doclength; i++)
			printf("%s\n",documentation[i]);
		exit (0);
	}

	/*
	 * check if X values given or incremented
	 */
	getpar_("x","s",string);
	sscanf(string,"%s",word);
	if( word[0] == 'i' || word[0] == 'I') increment=1;

	/*
	 * check if binary input format
	 */
	ascii=1; 
	getpar_("format","s",string);
	sscanf(string,"%s",word);
	if( word[0] == 'b' || word[0] == 'B') ascii=0;

	nplot=400; 
	getpar_("nplot","d",&nplot);

	strcpy(string, "graph.error.log");
/*
	if(getpar_("error","s",string))
	{
*/
		if((freopen(string,"w",stderr)) == NULL)
			fprintf(stderr,"cannot open error file %s\n", string);
	/*}*/


	if (getpar_("in","s",string)==0)
		in_file=stdin;
	else
	{
		sscanf(string,"%s",word);
		if ((in_file=fopen(word,"r"))==NULL)
		{
			fprintf (stderr,"cannot open in=%s\n",word);
			exit(-1);
		}
	}

	if (getpar_("out","s",string)==0)
		out_file=stdout;
	else
	{
		sscanf(string,"%s",word);
		if ((out_file=fopen(word,"w"))==NULL)
		{
			fprintf (stderr,"cannot open out=%s\n",word);
			exit(-1);
		}
	}

	logx = 0;
	getpar_("logx","d",&logx);
	logy = 0;
	getpar_("logy","d",&logy);
	loglog = 0;
	getpar_("loglog","d",&loglog);
	if (loglog)
	{
		logx = 1;
		logy = 1;
	}

	/*
	 * allocate space and read in the data
	 */
	errflag = 0;
	for (i=0; i<nplot; i++)
	{
		if(ascii)
		{
			if(fgets(string,80,in_file)==NULL) break;
			sscanf(string,"%d",npts+i);
			if (!*(npts+i))         /*  Skip lines which don't start with digits */
			{
				i--;
				continue;
			}
			if( npts[i] < 0 || npts[i] > 500000)
			{
				fprintf(stderr,"npts[i] = %d in data set %d\n",npts[i],i+1);
				fprintf(stderr,"can't handle that...bye!\n");
				exit(1);
			}
			points[i] = (Gpoint *)calloc(npts[i],sizeof(Gpoint));
			pptr=points[i];
			if (increment)
			{
				sscanf(string,"%*f %f %f",&x0,&dx);
				for (j=0; j<npts[i]; j++)
				{
					if(fgets(string,80,in_file)==NULL)
						read_error(i);
					sscanf(string,"%f",&(pptr->y));
					pptr->x = x0+j*dx;
					pptr++; 
				}
			}
			else
			{
				for (j=0; j<npts[i]; j++)
				{
					if(fgets(string,80,in_file)==NULL)
						read_error(i);
					sscanf(string,"%f %f",&(pptr->x),&(pptr->y));
					pptr++; 
				}
			}
		}
		else
		{
			if( (npts[i] = getw(in_file)) == EOF) break;
			if( npts[i] < 0 || npts[i] > 500000)
			{
				fprintf(stderr,"npts[i] = %d in data set %d\n",npts[i],i+1);
				fprintf(stderr,"can't handle that...bye!\n");
				exit(1);
			}
			farray = (float *)calloc(npts[i],sizeof(float));
			points[i] = (Gpoint *)calloc(npts[i],sizeof(Gpoint));
			if (increment)
			{
				fread(&x0,sizeof(float),1,in_file);
				fread(&dx,sizeof(float),1,in_file);
				if(fread(farray,sizeof(float),npts[i],in_file)
				    != npts[i]) read_error(i);
				pptr=points[i];
				fptr = farray;
				for (j=0; j<npts[i]; j++)
				{
					pptr->x = x0+j*dx;
					pptr->y = *fptr;
					pptr++;
					fptr++;
				}
			}
			else
			{
				if(fread(farray,sizeof(float),npts[i],in_file)
				    != npts[i]) read_error(i);
				pptr=points[i];
				fptr = farray;
				for (j=0; j<npts[i]; j++)
				{
					pptr->x = *fptr;
					pptr++;
					fptr++;
				}
				if(fread(farray,sizeof(float),npts[i],in_file)
				    != npts[i]) read_error(i);
				pptr=points[i];
				fptr = farray;
				for (j=0; j<npts[i]; j++)
				{
					pptr->y = *fptr;
					pptr++;
					fptr++;
				}
			}
			free(farray);
		}

		/*
		 * Take logarithms, if necessary
		 */
		pptr=points[i];
		for(j=0; j<npts[i]; j++)
		{
			if (logx)
			{
				if (pptr->x <= 0)
				{
				fprintf(stderr,"Input error: x <= 0 for plot %d, point %d\n",
					i+1,j+1);
				errflag = 1;
				}
				else pptr->x = log10(pptr->x);
			}
			if (logy)
			{
				if (pptr->y <= 0)
				{
				fprintf(stderr,"Input error: y <= 0 for plot %d, point %d\n",
					i+1,j+1);
				errflag = 1;
				}
				else pptr->y = log10(pptr->y);
			}
			pptr++; 
		}
	}
	if(errflag) exit(1);
	nplot=i;


	/*
	 * find the min and max values in the data
	 */
	window->xmin = points[0]->x;
	window->ymin = points[0]->y;
	window->xmax = points[0]->x;
	window->ymax = points[0]->y;
	for (i = 0; i < nplot; i++)
	{
		pptr=points[i];
		if (!increment)
		{
			for (j = 0; j < npts[i]; j++)
			{
				if(pptr->x < window->xmin) window->xmin = pptr->x;
				if(pptr->y < window->ymin) window->ymin = pptr->y;
				if(pptr->x > window->xmax) window->xmax = pptr->x;
				if(pptr->y > window->ymax) window->ymax = pptr->y;
				pptr++; 
			}
		}
		else
		{
			if(pptr->x < window->xmin) window->xmin = pptr->x;
			xf = (pptr + npts[i] -1)->x;
			if(xf > window->xmax) window->xmax = xf;
			for (j = 0; j < npts[i]; j++)
			{
				if(pptr->y < window->ymin) window->ymin = pptr->y;
				if(pptr->y > window->ymax) window->ymax = pptr->y;
				pptr++; 
			}
		}
	}

	if (logx)
	{
		window->xmin = floor((double)(window->xmin));
		window->xmax = ceil((double)(window->xmax));
	}

	if (logy)
	{
		window->ymin = floor((double)(window->ymin));
		window->ymax = ceil((double)(window->ymax));
	}

	/*
	 * set up plotting and numbering scales
	 * check for bad values for max, min on log axes
	 */
	errflag = 0;
	if(getpar_("xmin","f",&xmin))
	{
		if(logx)
		{
			if( xmin <=0 )
			{
				fprintf(stderr,"Input error: xmin <= 0");
				errflag = 1;
			}
			else
				xmin = log10((double)xmin);
		}
		window->xmin = xmin;
	}
	if(getpar_("ymin","f",&ymin))
	{
		if(logy)
		{
			if( ymin <=0 )
			{
				fprintf(stderr,"Input error: ymin <= 0");
				errflag = 1;
			}
			else
				ymin = log10((double)ymin);
		}
		window->ymin = ymin;
	}
	if(getpar_("xmax","f",&xmax))
	{
		if(logx)
		{
			if( xmax <=0 )
			{
				fprintf(stderr,"Input error: xmax <= 0");
				errflag = 1;
			}
			else
				xmax = log10((double)xmax);
		}
		window->xmax = xmax;
	}
	if(getpar_("ymax","f",&ymax))
	{
		if(logy)
		{
			if( ymax <=0 )
			{
				fprintf(stderr,"Input error: ymax <= 0");
				errflag = 1;
			}
			else
				ymax = log10((double)ymax);
		}
		window->ymax = ymax;
	}

	if((window->xmax - window->xmin) <= 0)
	{
		fprintf(stderr,"Input error: xmin <= xmax\n");
		errflag = 1;
	}

	if((window->ymax - window->ymin) <= 0)
	{
		fprintf(stderr,"Input error: ymin <= ymax\n");
		errflag = 1;
	}

	if(errflag) exit(1);

	xinch=8.0; 
	getpar_("xinch","f",&xinch);
	yinch=4.0; 
	getpar_("yinch","f",&yinch);
	if(loglog)
	{
		dummy = window->ymin + (window->xmax - window->xmin) * yinch/xinch;
		if( dummy > window->ymax )
		{
			window->ymax = dummy;
		}
		else
		{
			window->xmax = window->xmin +
				(window->ymax - window->ymin) * xinch/yinch;
		}
	}

	getscl (window->xmin,window->xmax,(int)(xinch*0.75),&dummy,&dummy,&dxbuf);
	getpar_("dxnum","f",&dxbuf);
	if(dxbuf > 0.) dxnum = dxbuf;
	else
	{
		fprintf (stderr,"input error: dxnum <= 0\n");
		exit(-1);
	}
	for (x0num=(int)((window->xmin)/dxnum)*dxnum-dxnum; x0num<(window->xmin);
				x0num+=dxnum);
	getpar_("x0num","f",&x0num);

	getscl (window->ymin,window->ymax,(int)(yinch*0.75),&dummy,&dummy,&dybuf);
	getpar_("dynum","f",&dybuf);
	if(dybuf > 0.) dynum = dybuf;
	else
	{
		fprintf (stderr,"input error: dynum <= 0\n");
		exit(-1);
	}

	for (y0num=(int)((window->ymin)/dynum)*dynum-dynum; y0num<(window->ymin);
				y0num+=dynum);
	getpar_("y0num","f",&y0num);
	nxtic = 1; 
	getpar_("nxtic","d",&nxtic);
	nytic = 1; 
	getpar_("nytic","d",&nytic);
	numsz=0.20; 
	getpar_("numsz","f",&numsz);
	numfat = 0; 
	getpar_("numfat","d",&numfat);
	dxtic = dxnum/nxtic;
	for (x0tic=x0num-nxtic*dxtic; x0tic<(window->xmin); x0tic+=dxtic);
	dytic = dynum/nytic;
	for (y0tic=y0num-nytic*dytic; y0tic<(window->ymin); y0tic+=dytic);

	/*
	 * set up plotting symbols and colors
	 */
	for (i=0; i<NPMAX; i++)
	{
		symbol[i] = 0;
		symbsc[i] = 1.0;
		plotcol[i] = 6-i%6;
		plotfat[i] = 0;
		linetype[i] = 1;
	}
	if(getpar_("symbol","s",string))
	{
		i=0;
		sptr = string;
		while(*sptr)
		{
			if( *sptr == '\\')
			{
				sptr++;
				symbol[i] = *sptr - '0';
			}
			else if((*sptr <= ' ')||(*sptr == '\0177'))
			{
				symbol[i] = 0;
			}
			else symbol[i] = *sptr;
			i++; sptr++;
			if( i == NPMAX) break;
		}
	}
	getpar_("symbscale","f",symbsc);
	getpar_("plotcol","d",plotcol);
	getpar_("plotfat","d",plotfat);
	getpar_("linetype","d",linetype);

	/*
	 * setup axes grid and labels
	 */
	grid=0;  
	getpar_("grid","d",&grid);
	gridcol[0] = 7; 
	gridcol[1] = 7;  
	getpar_("gridcol","d",gridcol);
	gridfat[0] = 0; 
	gridfat[1] = 0;  
	getpar_("gridfat","d",gridfat);
	box=1;  
	getpar_("box","d",&box);
	axiscol[0] = 7; 
	axiscol[1] = -5; 
	getpar_("axiscol","d",axiscol);
	if (axiscol[1] == -5) axiscol[1] = axiscol[0];
	numcol = axiscol[0]; 
	getpar_("numcol","d",&numcol);
	axisfat[0] = 0; 
	axisfat[1] = -5; 
	getpar_("axisfat","d",axisfat);
	if (axisfat[1] == -5) axisfat[1] = axisfat[0];

	strcpy (xlabel," "); 
	getpar_("xlabel","s",xlabel);
	strcpy (ylabel," "); 
	getpar_("ylabel","s",ylabel);
	strcpy (title," "); 
	getpar_("title","s",title);
	lablsz=0.2; 
	getpar_("lablsz","f",&lablsz);
	titlsz=0.3; 
	getpar_("titlsz","f",&titlsz);
	titlcol = 7; 
	getpar_("titlcol","d",&titlcol);
	titlfat = 0; 
	getpar_("titlfat","d",&titlfat);
	lablcol = 7; 
	getpar_("lablcol","d",&lablcol);
	lablfat = 0; 
	getpar_("lablfat","d",&lablfat);


	/*
	 * initialization for plotting
	 */
	typename = ggetwstype(out_file);
		if(((wstype = gwstype(typename)) == 0) && !(piped_out))
		{
			fprintf(stderr, "graph: Unknown workstation type.\n");
			fprintf(stderr, "       Set the WSTYPE environment variable,");
			fprintf(stderr, " or redirect your output.\n");
			exit(1);
		}
		if(wstype == VPLOT) fontprec->prec = GSTROKE;
	else fontprec->prec = GSTRING;
	gopengks(1000,stderr);
	gopenws(WORKSTATION,out_file,typename);
	gactivatews(WORKSTATION);

	/*
	 * figure out longest Y axis number for origin setting
	 */
	y_start = (int)(window->ymin);
	y_end = (int)(window->ymax);
	nychrmax = 0;
	if (logy) {
		y0num = (float)(y_start);
		ntic = y_end - y_start + 1;
		dynum = 1.;
	}
	else {
		ntic = (int)(((window->ymax - y0num)/dynum)) + 1.;
	}
	ynum = y0num;
	while(ntic--)
	{
		if (fabs(ynum)<(window->ymin - window->ymax)/10000) ynum=0.0;
		if (logy)
			nchr = cntnum(string,pow(10.,ynum));
		else nchr = cntnum(string,ynum);
		if ( nchr > nychrmax ) nychrmax = nchr;
		ynum += dynum;
	}

	/*
	 * try to center in 8.5x11 page 
	 * if can't center, just leave 0.3 inch on lower left
	 */
		if ((getpar_("xorigin","f",&xorig)) == 0)
		{
		xneeded =  (2.0 * lablsz) + ((nychrmax+2) * numsz);
		xorig = 0.3 + xneeded + (10.4 - xinch - xneeded)/2.0;
		if(xorig < (xneeded + 0.3)) xorig=xneeded + 0.3;
		}
		if ((getpar_("yorigin","f",&yorig)) == 0)
		{
		yneeded =  2. * (numsz + lablsz);
		yorig = 0.3 + yneeded + (7.9 - yinch - yneeded)/2.0;
		if(yorig < (yneeded + 0.3)) yorig=yneeded + 0.3;
		}

	page->xmin = 0.;
	page->ymin = 0.;
	page->xmax = (xinch + 2. * xorig);
	page->ymax = (yinch + 2. * yorig);
	pagelength = (page->xmax > page->ymax) ? page->xmax : page->ymax;
	page->xmax /= pagelength;
	page->ymax /= pagelength;
	gsetwswindow(WORKSTATION,page);

	ginqmaxdisplaysize(typename,&dcunits,&dcsize,&rastersize);
	/*
	 * if possible do absolute scaling
	 */
	if(dcunits == GDC_METERS)
	{
		page->xmax *= (pagelength * .0254);
		page->ymax *= (pagelength * .0254);
		gsetwsviewport(WORKSTATION,page);
	}

	page->xmax = xinch;
	page->ymax = yinch;
	gsetwindow(INCH,page);

	page->xmin = xorig/pagelength;
	page->ymin = yorig/pagelength;
	page->xmax = (xinch + xorig)/pagelength;
	page->ymax = (yinch + yorig)/pagelength;
	gsetviewport(INCH,page);
	gsetviewport(USER,page);

	gsetwindow(USER,window);
	xscale = xinch/(window->xmax - window->xmin);
	yscale = yinch/(window->ymax - window->ymin);

	gselntran(INCH);
	gsetclip(GNOCLIP);

	/*
	 * establish a reasonable tick size
	 */
	ticlength = ((xinch < yinch) ? xinch : yinch)/30.;
	if(ticlength < 0.10) ticlength = 0.10;

	/*
	 * plot and label y-axis
	 */
	tick1[0].x = 0.;

	/*
	 * first do minor grid or ticks
	 */
	if(grid < 2)
	{
		gsetlinewidth((float)(axisfat[1]+1));
		gsplci(axiscol[1]);
		tick1[1].x = ticlength/2.;
		tick2[1].x = xinch - ticlength/2.;
		tick2[0].x = xinch;
	}
	else
	{
		gsetlinewidth((float)(gridfat[1]+1));
		gsplci(gridcol[1]);
		tick1[1].x = xinch;
	}

	/*
	 * if logy do log cycles on y axis
	 */
	if(logy) 
	{
		cycle_index = 0;
		cycle_range = abs(y_end - y_start);
		ytic = (float)(y_start);
		for (y_tic=1; ; )
		{
			ytic = log10((float)(y_tic)) + (float)(cycle_index);
			if (ytic > cycle_range) break;
			ytic = ytic + y_start;
			tick1[0].y = tick1[1].y = (ytic - window->ymin) * yscale;
			if(y_tic != 1) gpolyline(2,tick1);
			if(box && (grid < 2))
			{
				tick2[0].y = tick2[1].y = tick1[0].y;
				if(y_tic != 1) gpolyline(2,tick2);
			}
			y_tic++;
			if (y_tic == 10)
			{
				cycle_index++;
				y_tic = 1;
			}
		}
	}
	else
	{
		ntic = (int)(((window->ymax - y0tic)/dytic)) + 1.;
		ytic = y0tic;
		while(ntic--)
		{
			tick1[0].y = tick1[1].y = (ytic - window->ymin) * yscale;
			gpolyline(2,tick1);
			if(box && (grid < 2))
			{
				tick2[0].y = tick2[1].y = tick1[0].y;
				gpolyline(2,tick2);
			}
			ytic += dytic;
		}
	}

	/*
	 * now do numbered grid or ticks
	 */
	gstxci(numcol);
	fontprec->font = numfat + 1;
	gsettextfontprec(fontprec);
	align->hor = GTH_RIGHT; align->ver = GTV_HALF;
	gsettextalign(align);
	gsetcharheight(numsz);
	vec->x = -(numsz);
	if(grid)
	{
		gsetlinewidth((float)(gridfat[0]+1));
		gsplci(gridcol[0]);
		tick1[1].x = xinch;
	}
	else
	{
		gsetlinewidth((float)(axisfat[1]+1));
		gsplci(axiscol[1]);
		tick1[1].x = ticlength;
		tick2[1].x = xinch - ticlength;
		tick2[0].x = xinch;
	}
	if (logy) {
		y0num = (float)(y_start);
		ntic = y_end - y_start + 1;
		dynum = 1.;
	}
	else {
		ntic = (int)(((window->ymax - y0num)/dynum)) + 1.;
	}
	ynum = y0num;
	while(ntic--)
	{
		if (fabs(ynum)<(window->ymin - window->ymax)/10000) ynum=0.0;
		if (logy)
			nchr = cntnum(string,pow(10.,ynum));
		else nchr = cntnum(string,ynum);
		vec->y = tick1[0].y = tick1[1].y = (ynum - window->ymin) * yscale;
		gpolyline(2,tick1);
		if(box && (!grid))
		{
			tick2[0].y = tick2[1].y = tick1[0].y;
			gpolyline(2,tick2);
		}
		if (numsz > 0) gtext(vec,string);
		ynum += dynum;
	}

	/*
	 * do y label
	 */
	vec->x = -1.; vec->y = 0.;
	fontprec->font = lablfat + 1;
	gsettextfontprec(fontprec);
	gsetcharup(vec);
	align->hor = GTH_CENTER; align->ver = GTV_BOTTOM;
	gsettextalign(align);
	gsetcharheight(lablsz);
	gstxci(lablcol);
	vec->y = yinch/2.; 
	vec->x = -(((nychrmax+1) * numsz) + lablsz);
	if (lablsz > 0) gtext(vec,ylabel);
	vec->x = 0.; vec->y = 1.;
	gsetcharup(vec);

	/*
	 *  plot and label x-axis
	 */
	tick1[0].y = 0;

	/*
	 * first do minor grid or ticks
	 */
	if(grid < 2)
	{
		gsetlinewidth((float)(axisfat[1]+1));
		gsplci(axiscol[1]);
		tick1[1].y = ticlength/2.;
		tick2[1].y = yinch - ticlength/2.;
		tick2[0].y = yinch;
	}
	else
	{
		gsetlinewidth((float)(gridfat[1]+1));
		gsplci(gridcol[1]);
		tick1[1].y = yinch;
	}

	/*
	 * if logx do log cycles on x axis
	 */
	if (logx) 
	{
		x_start = (int)(window->xmin);
		x_end = (int)(window->xmax);
		cycle_index = 0;
		cycle_range = abs(x_end - x_start);
		xtic = (float)(x_start);
		for (x_tic=1; ; )
		{
			xtic = log10((float)(x_tic)) + (float)(cycle_index);
			if (xtic > cycle_range) break;
			xtic = xtic + x_start;
			tick1[0].x = tick1[1].x = (xtic - window->xmin) * xscale;
			if(x_tic != 1) gpolyline(2,tick1);
			if(box && (grid < 2))
			{
				tick2[0].x = tick2[1].x = tick1[0].x;
				if(x_tic != 1) gpolyline(2,tick2);
			}
			x_tic++;
			if (x_tic == 10)
			{
				cycle_index++;
				x_tic = 1;
			}
		}
	}
	else
	{
		ntic = (int)(((window->xmax - x0tic)/dxtic)) + 1.;
		xtic = x0tic;
		while(ntic--)
		{
			tick1[0].x = tick1[1].x = (xtic - window->xmin) * xscale;
			gpolyline(2,tick1);
			if(box && (grid < 2))
			{
				tick2[0].x = tick2[1].x = tick1[0].x;
				gpolyline(2,tick2);
			}
			xtic += dxtic;
		}
	}

	/*
	 * now do numbered grid or ticks
	 */
	gstxci(numcol);
	fontprec->font = numfat + 1;
	gsettextfontprec(fontprec);
	align->hor = GTH_CENTER; align->ver = GTV_CAP;
	gsettextalign(align);
	gsetcharheight(numsz);
	vec->y = -numsz;
	if(grid)
	{
		gsetlinewidth((float)(gridfat[0]+1));
		gsplci(gridcol[0]);
		tick1[1].y = yinch;
	}
	else
	{
		gsetlinewidth((float)(axisfat[1]+1));
		gsplci(axiscol[1]);
		tick1[1].y = ticlength;
		tick2[1].y = yinch - ticlength;
		tick2[0].y = yinch;
	}
	if (logx) {
		x0num = (float)(x_start);
		ntic = x_end - x_start + 1;
		dxnum = 1.;
	}
	else {
		ntic = (int)(((window->xmax - x0num)/dxnum)) + 1.;
	}
	xnum = x0num;
	while(ntic--)
	{
		if (fabs(xnum)<(window->xmin - window->xmax)/10000) xnum=0.0;
		if (logx)
			nchr = cntnum(string,pow(10.,xnum));
		else nchr = cntnum(string,xnum);
		vec->x = tick1[0].x = tick1[1].x = (xnum - window->xmin) * xscale;
		gpolyline(2,tick1);
		if(box && (!grid))
		{
			tick2[0].x = tick2[1].x = tick1[0].x;
			gpolyline(2,tick2);
		}
		if (numsz > 0) gtext(vec,string);
		xnum += dxnum;
	}

	/*
	 * do x label
	 */
	align->hor = GTH_CENTER; align->ver = GTV_TOP;
	gsettextalign(align);
	fontprec->font = lablfat + 1;
	gsettextfontprec(fontprec);
	gsetcharheight(lablsz);
	gstxci(lablcol);
	vec->x = xinch/2.; 
	vec->y = -(lablsz + (2. * numsz));
	if (lablsz > 0) gtext(vec,xlabel);

	/*
	 * plot title
	 */
	gsetcharheight(titlsz);
	fontprec->font = titlfat + 1;
	gsettextfontprec(fontprec);
	align->hor = GTH_CENTER; align->ver = GTV_BOTTOM;
	gsettextalign(align);
	gstxci(titlcol);
	vec->y = yinch + titlsz;
	if (titlsz > 0 ) gtext(vec,title);

	/*
	 * draw nplot plots
	 */
	gselntran(USER);
	gsetclip(GCLIP);
	for (i=0; i<nplot; i++)
	{
		pptr=points[i];
		if( symbol[i])
		{
			gspmci(plotcol[i]);
			gsetmarkertype(symbol[i]);
			gsetmarkersize(symbsc[i]);
			gpolymarker(npts[i],points[i]);
		}
		else
		{
			gsplci(plotcol[i]);
			gsetlinetype(linetype[i]);
			gsetlinewidth((float)(plotfat[i]+1));
			gpolyline(npts[i],points[i]);
		}
	}

	/*
	 * draw plot frame
	 */
	gselntran(INCH);
	gsetclip(GNOCLIP);
	gsplci(axiscol[0]);
	gsetlinetype(1);
	gsetlinewidth((float)(axisfat[0]+1));
	tick1[0].x = 0.; tick1[0].y = 0.;
	tick1[1].x = 0.; tick1[1].y = yinch;
	gpolyline(2,tick1);
	tick1[1].x = xinch; tick1[1].y = 0.;
	gpolyline(2,tick1);
	if(box)
	{
		tick1[0].x = xinch; tick1[0].y = yinch;
		gpolyline(2,tick1);
		tick1[1].x = 0.; tick1[1].y = yinch;
		gpolyline(2,tick1);
	}
		if(wstype != VPLOT) greqloc(WORKSTATION, 1, &dummy2);
#ifdef DEBUG
	fprintf(stderr, "xorigin=%f yorigin=%f lablsz=%f numsz=%f titlsz=%f",
		xorig, yorig, lablsz, numsz, titlsz);
#endif DEBUG
	gdeactivatews(WORKSTATION);
	gclosews(WORKSTATION);
	exit(0);
}
